#include "windows.h"

#include <maya/MPxFileTranslator.h>
#include <maya/MFnDagNode.h>
#include <maya/MItDependencyNodes.h>
#include <maya/MFnSkinCluster.h>

#include <MDt.h>
#include <MDtExt.h>

#include <rpmatfx.h>
#include <rpcollis.h>
#include <rpanim.h>
#include <rphanim.h>
#include <rpmorph.h>
#include <rpskin.h>
#include <rpspline.h>
#include <rpmipkl.h>

#include "rwcommon.h"
#include "global.h"

bool OpenRenderWare(HWND window, bool openRpAnim, bool openRpSkin, bool openRpHAnim,
               bool openCollision, bool openMorph, bool openSpline,
               bool openMipKL, bool openUserData)
{
    RwEngineOpenParams openParams;

    RwEngineInit(NULL, 0);

    if (!RpWorldPluginAttach())
    {
        return false;
    }

    if (!RpMatFXPluginAttach())
    {
        return false;
    }

    if (openRpAnim)
    {
        if (!RpAnimPluginAttach())
        {
            return false;
        }
    }

    if (openRpHAnim)
    {
        if (!RpHAnimPluginAttach())
        {
            return false;
        }
    }

    if (openRpSkin)
    {
        if (!RpSkinPluginAttach())
        {
            return false;
        }
    }

    if (openCollision)
    {
        if (!RpCollisionPluginAttach())
        {
            printf("BSP collision plugin failed to open!\n");
            return false;
        }
    }

    if (openMorph)
    {
        if (!RpMorphPluginAttach())
        {
            printf("Morph plugin failed to open!\n");
            return false;
        }
    }

    if (openSpline)
    {
        if (!RpSplinePluginAttach())
        {
            printf("Spline plugin failed to open!\n");
            return false;
        }
    }

    if (openMipKL)
    {
        if (!RpMipmapKLPluginAttach())
        {
            printf("MipmapKL plugin failed to open!\n");
            return false;
        }
    }

    if (openUserData == true)
    {
        if (!RpUserDataPluginAttach())
        {
            printf("UserData plugin failed to open!\n");
            return false;
        }
    }

    openParams.displayID = (void *)window;

    if (!RwEngineOpen(&openParams))
    {
        RwEngineTerm();
        return false;
    }
    RwEngineSetSubSystem(0);

    RwResourcesSetArenaSize(1024 * 1024 * 2);

    if (!RwEngineStart())
    {
        RwEngineClose();
        RwEngineTerm();
    
        return false;
    }

    switch(globalData->m_triStripper)
    {
        case TRISTRIP_DEFAULT:
        default:
            if (!RpMeshSetTriStripMethod(NULL, NULL))
            {
                return false;
            }
            break;
        case TRISTRIP_PREPROCESS:
            if (!RpMeshSetTriStripMethod(RpBuildMeshGeneratePreprocessTriStrip, NULL))
            {
                return false;
            }
            break;
        case TRISTRIP_EXHAUSTIVE:
            if (!RpMeshSetTriStripMethod(RpBuildMeshGenerateExhaustiveTriStrip, NULL))
            {
                return false;
            }
            break;
    }
    
    return true;
}

bool CloseRenderWare(void)
{
    RwEngineStop();
    RwEngineClose();
    RwEngineTerm();
    return true;
}

/* Material Color functions */
RpMaterial *CheckMaterialColors(RpMaterial *material, void *pData)
{
    bool            *colored = (bool *)pData;
    const RwRGBA    *color;

    color = RpMaterialGetColor(material);
    if (color->red      != 255 ||
        color->green    != 255 ||
        color->blue     != 255 ||
        color->alpha    != 255)
    {
        *colored = true;
        return NULL;
    }
    
    return material;
}

bool GeometryHasColoredMaterials(RpGeometry *geometry)
{
    bool colored = false;

    RpGeometryForAllMaterials(geometry, CheckMaterialColors, &colored);

    return colored;
}

bool WorldHasColoredMaterials(RpWorld *world)
{
    bool colored = false;

    RpWorldForAllMaterials(world, CheckMaterialColors, &colored);

    return colored;
}

/* Material FX functions */
RpMaterial *CheckMaterialFX(RpMaterial *material, void *pData)
{
    bool *hasFX = (bool *)pData;

    if (RpMatFXMaterialGetEffects(material) != rpMATFXEFFECTNULL)
    {
        *hasFX = true;
        return NULL;
    }

    return material;
}

bool GeometryHasMaterialFX(RpGeometry *geometry)
{
    bool hasFX = false;

    RpGeometryForAllMaterials(geometry, CheckMaterialFX, &hasFX);

    return hasFX;
}

RpMesh *
MeshCheckMaterialFX(RpMesh *mesh, RpMeshHeader *meshHeader, void *pData)
{
    if (CheckMaterialFX(mesh->material, pData) == NULL)
    {
        return NULL;
    }

    return mesh;
}

RpWorldSector *
WorldSectorHasMaterialFX(RpWorldSector *sector, void *pData __RWUNUSED__)
{
    bool hasFX = false;

    RpWorldSectorForAllMeshes(sector, MeshCheckMaterialFX, &hasFX);

    if (hasFX)
    {
        RpMatFXWorldSectorEnableEffects(sector);
    }

    return sector;
}


void forAllNodesInScene(MFn::Type fnType, MayaObjectCallBack callBack, void *pData)
{
    MItDependencyNodes iter(MFn::kInvalid);
    for (; !iter.isDone(); iter.next()) 
    {
        MFn::Type   objType;
        MObject     object = iter.item();
        
        objType = object.apiType();
        
        if (object.apiType() == fnType)
        {
            if (callBack(object, pData) == MObject(MObject::kNullObj))
            {
                return;
            }
        }
    }

    return;
}

void OffsetTextureCoords(RwV3d *uv, int uvCount)
{
    float   minU = 0.0f, maxV = 1.0f;
    float   uOffset, vOffset;
    int     i;
    
    for (i = 0; i < uvCount; i++)
    {
        if (uv[i].x < minU)
        {
            minU = uv[i].x;
        }
        if (uv[i].y > maxV)
        {
            maxV = uv[i].y;
        }
    }

    uOffset = (float)(-floor(minU));

    if (maxV == floor(maxV))
    {
        vOffset = maxV;
    }
    else
    {
        vOffset = (float)(maxV + (1 - (maxV - floor(maxV))));
    }

    for (i = 0; i < uvCount; i++)
    {
        uv[i].x = uv[i].x + uOffset;
        uv[i].y = vOffset - uv[i].y;
    }
}